import asyncio
import json
import time

from pylog.pylogger import PyLogger

from py_pli.pylib import GlobalVar
from py_pli.pylib import VUnits

from py_pli.pyexception import UrpcFirmwareException

from virtualunits.vu_measurement_unit import VUMeasurementUnit

from fleming.common.meas_seq_generator_base import meas_seq_generator
from fleming.common.meas_seq_generator_base import OutputSignal
from fleming.common.meas_seq_generator_base import TriggerSignal
from fleming.common.meas_seq_generator_base import MeasurementChannel
from fleming.common.meas_seq_generator_base import IntegratorMode
from fleming.common.meas_seq_generator_base import AnalogControlMode

from urpc_enum.barcodereaderparameter import BarcodeReaderParameter
from urpc_enum.dualmoverparameter import DualMoverParameter
from urpc_enum.fancontrolparameter import FanControlParameter
from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.serialparameter import SerialParameter

from urpc_enum.error import MeasurementErrorCode

from fleming.common.firmware_util import *
from fleming.common.node_io import *


signals = {
    'none'      : (0),
    'flash'     : (1 << 23),
    'alpha'     : (1 << 22),
    'ingate2'   : (1 << 17),
    'ingate1'   : (1 << 16),
    'hvgate2'   : (1 << 13),
    'hvgate1'   : (1 << 12),
    'hvon3'     : (1 << 10),
    'hvon2'     : (1 <<  9),
    'hvon1'     : (1 <<  8),
    'rstaux'    : (1 <<  6),
    'rstabs'    : (1 <<  5),
    'rstref'    : (1 <<  4),
    'rstpmt2'   : (1 <<  1),
    'rstpmt1'   : (1 <<  0),
}

triggers = {
    'trf'   : (1 << 8),
    'aux'   : (1 << 6),
    'abs'   : (1 << 5),
    'ref'   : (1 << 4),
    'pmt3'  : (1 << 2),
    'pmt2'  : (1 << 1),
    'pmt1'  : (1 << 0),
}

pd_channel = {
    'pd1' : MeasurementChannel.PD1,
    'pd2' : MeasurementChannel.PD2,
    'pd3' : MeasurementChannel.PD3,
    'pd4' : MeasurementChannel.PD4,
}

pd_sample = {
    'pd1' : TriggerSignal.SamplePD1,
    'pd2' : TriggerSignal.SamplePD2,
    'pd3' : TriggerSignal.SamplePD3,
    'pd4' : TriggerSignal.SamplePD4,
}

pd_reset = {
    'pd1' : OutputSignal.Deprc_IntRstPD1,
    'pd2' : OutputSignal.Deprc_IntRstPD3,
    'pd3' : OutputSignal.Deprc_IntRstPD2,
    'pd4' : OutputSignal.Deprc_IntRstPD4,
}

integrator_mode = {
    'reset'     : IntegratorMode.full_reset,
    'low_reset' : IntegratorMode.low_range_reset,
    'auto'      : IntegratorMode.integrate_autorange,
    'fixed'     : IntegratorMode.integrate_with_fixed_range,
    'low'       : IntegratorMode.integrate_in_low_range,
    'high'      : IntegratorMode.integrate_in_high_range,
}

ANALOGCONTROLMODE_ADD_TO_HIGH_OFFSET            = 1
ANALOGCONTROLMODE_ADD_TO_LOW_OFFSET             = 2
ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET    = 3

OUTPUTSIGNAL_ALPHA_HTS = 1 << 21     


async def set_alpha_excitation(enable):
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    op_id = 'al_ex'
    seq_gen = meas_seq_generator()
    if enable:
        seq_gen.SetSignals(OutputSignal.Alpha)      # Set Signal to turn Laser On!!!!
    else:
        seq_gen.ResetSignals(OutputSignal.Alpha)    # Reset Signal to turn Laser Off!!!
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await measurement_unit.ExecuteMeasurement(op_id)


async def serial_loopback(serial_name, *txdata) -> bytes:
    serial = get_serial_endpoint(serial_name)
    await serial.SetParameter(SerialParameter.ReadTimeout, 1900)
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(txdata), bytes(txdata), timeout=2)
    rxdata = None
    try:
        response = await serial.Read(len(txdata), timeout=2)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    return rxdata


async def bcr_loopback(bcr_name, *txdata) -> bytes:
    bcr = get_barcode_reader_endpoint(bcr_name)
    
    await bcr.SetParameter(BarcodeReaderParameter.ReadTimeout, 2000)
    await bcr.SelectChannel(barcode_reader_endpoints[bcr_name]['channel'])
    await bcr.ClearReadBuffer(timeout=1)
    
    await bcr.Write(len(txdata), bytes(txdata), timeout=2)
    rxdata = None
    try:
        response = await bcr.Read(len(txdata), timeout=2)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    return rxdata


async def seq_trf_loopback():
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers['trf'],   # Set Trigger Output
        0x74000000 | 1,                 # Wait For Trigger Input
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    await asyncio.sleep(0.1)
    if (await meas.GetStatus())[0] & 0x01:
        return f"seq_trf_loopback() passed"
    else:
        return f"seq_trf_loopback() failed"


async def seq_set_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x02000000 | signals[name],     # Set Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    
    return f"seq_set_signal() done"


async def seq_clear_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x03000000 | signals[name],     # Reset Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    
    return f"seq_clear_signal() done"


async def eef_test_script():

    #Test script for EEF
    #This script checks the following interfaces:

        #TRF Serial TRG and Syn
        #TMP1 & TMP2
        #BCR1, BCR2 BCREXT Serial and IO
        #M1 to M4 with ENC
        #ALD Excitation and NTC
        #IL
        #XFL
        #EXP
        #REF 1 to 4
        #USL Fan
        #TEC1, TEC2 and TEC RES
        #FAN1 and FAN2
        #PD REF, ABS, AUX and PD4 (need to uncomment PD4 Test for EEF >= rev8)
        #PMT1 and PMT2  IO and I2C
        #PMT1, PMT2 and USL Counting
        #HTS Alpha Power and Excitation

        #USB interface is tested prior during programming of MC6.


    fmb = get_node_endpoint('fmb')

    # Reset the ground control stop button status at the beginning of the test.
    GlobalVar.set_stop_gc(False)
    # Send information and test results to ground control, that will be printed in the output text box.

    eeftest_error = 0

    await start_firmware('fmb')
    await fmb.SetDigitalOutput(FMBDigitalOutput.PSUON, 0)       # Base Tester Power = ON (active low)

    await send_gc_msg(f"Start EEF Test", log=True)

    try:
        await asyncio.sleep(1)

        #TRF Interface Loopback Test
        await start_firmware('eef')

        txmsg = 'Hello World'
        rxdata = await serial_loopback('trf', *(txmsg.encode('utf-8')))
        rxmsg = bytes(rxdata).decode('utf-8')

        PyLogger.logger.info(f"TRF Loopback: '{rxmsg}'")

        if (txmsg != rxmsg):
            eeftest_error = 1
            await send_gc_msg(f"FAILED TRF Serial Loopback Test", log=True)
        else:
            await send_gc_msg(f"PASSED TRF Serial Loopback Test", log=True)

        trg_sync_loopback = (await seq_trf_loopback()) == 'seq_trf_loopback() passed'

        if not trg_sync_loopback:
            eeftest_error = 1
            await send_gc_msg(f"FAILED TRF Sync Loopback Test", log=True)
        else:
            await send_gc_msg(f"PASSED TRF Sync Loopback Test", log=True)

        #PMT Temp Sensor Test

        await start_firmware('eef')
        eef = get_node_endpoint('eef')

        temp1 = (await eef.GetAnalogInput(EEFAnalogInput.PMT1TEMPB))[0] * 256
        PyLogger.logger.info(f"PMT Temperature Sensor 1: {temp1:.2f} °C")
        if temp1 < 5 or temp1 > 50:
            eeftest_error = 1
            await send_gc_msg(f"FAILED TMP1 Test", log=True)
        else:
            await send_gc_msg(f"PASSED TMP1 Test", log=True)

        temp2 = (await eef.GetAnalogInput(EEFAnalogInput.PMT2TEMPB))[0] * 256
        PyLogger.logger.info(f"PMT Temperature Sensor 2: {temp2:.2f} °C")
        if temp2 < 5 or temp2 > 50:
            eeftest_error = 1
            await send_gc_msg(f"FAILED TMP2 Test", log=True)
        else:
            await send_gc_msg(f"PASSED TMP2 Test", log=True)

        #BCR1 Test
        await start_firmware('eef')

        txmsg = 'Hello World'
        rxdata = await bcr_loopback('bcr1', *(txmsg.encode('utf-8')))
        rxmsg = bytes(rxdata).decode('utf-8')

        PyLogger.logger.info(f"BCR1 Loopback: '{rxmsg}'")

        if (txmsg != rxmsg):
            eeftest_error = 1
            await send_gc_msg(f"FAILED BCR1 Loopback Test", log=True)     
        else:
            await send_gc_msg(f"PASSED BCR1 Loopback Test", log=True)     

        await eef.SetDigitalOutput(3,1) #set ntrig1
        await eef.SetDigitalOutput(2,1) #set nrst

        led_bcr1_on = (await eef.GetDigitalInput(1))[0] #Test BCR1 Input

        await eef.SetDigitalOutput(3,0) #reset ntrig1
        await eef.SetDigitalOutput(2,0) #reset nrst

        led_bcr1_off = (await eef.GetDigitalInput(1))[0] #Test BCR1 Input

        PyLogger.logger.info(f"led_bcr1_on  : {led_bcr1_on}")
        PyLogger.logger.info(f"led_bcr1_off : {led_bcr1_off}")

        if led_bcr1_on != 1 or led_bcr1_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED BCR1 IO Test", log=True)     
        else:
            await send_gc_msg(f"PASSED BCR1 IO Test", log=True)     

        #BCR2 Test
        txmsg = 'Hello World'
        rxdata = await bcr_loopback('bcr2', *(txmsg.encode('utf-8')))
        rxmsg = bytes(rxdata).decode('utf-8')

        PyLogger.logger.info(f"BCR2 Loopback: '{rxmsg}'")

        if (txmsg != rxmsg):
            eeftest_error = 1
            await send_gc_msg(f"FAILED BCR2 Loopback Test", log=True)
        else:
            await send_gc_msg(f"PASSED BCR2 Loopback Test", log=True)

        await eef.SetDigitalOutput(4,1) #set ntrig1
        await eef.SetDigitalOutput(2,1) #set nrst

        led_bcr2_on = (await eef.GetDigitalInput(2))[0] #Test BCR2 Input

        await eef.SetDigitalOutput(4,0) #reset ntrig2
        await eef.SetDigitalOutput(2,0) #reset nrst

        led_bcr2_off = (await eef.GetDigitalInput(2))[0] #Test BCR2 Input

        PyLogger.logger.info(f"led_bcr2_on  : {led_bcr2_on}")
        PyLogger.logger.info(f"led_bcr2_off : {led_bcr2_off}")

        if led_bcr2_on != 1 or led_bcr2_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED BCR2 IO Test", log=True)
        else:
            await send_gc_msg(f"PASSED BCR2 IO Test", log=True)

        #BCREXT Test
        txmsg = 'Hello World'
        rxdata = await bcr_loopback('bcr3', *(txmsg.encode('utf-8')))    #seems to be wrong because same id as bcr2
        rxmsg = bytes(rxdata).decode('utf-8')

        PyLogger.logger.info(f"BCRRXT Loopback: '{rxmsg}'")

        if (txmsg != rxmsg):
            eeftest_error = 1
            await send_gc_msg(f"FAILED BCREXT Loopback Test", log=True)
        else:
            await send_gc_msg(f"PASSED BCREXT Loopback Test", log=True)

        await eef.SetDigitalOutput(5,1) #set extntrig
        await eef.SetDigitalOutput(2,1) #set nrst
        await eef.SetDigitalOutput(6,1) #set bcr ch3
        await eef.SetDigitalOutput(7,1) #set bcr ch4
        await eef.SetDigitalOutput(8,1) #set bcr ch5

        led_bcr_ext_on = (await eef.GetDigitalInput(3))[0] #Test BCREXT Input

        await eef.SetDigitalOutput(5,0) #reset extntrig
        await eef.SetDigitalOutput(2,0) #reset nrst
        await eef.SetDigitalOutput(6,0) #reset bcr ch3
        await eef.SetDigitalOutput(7,0) #reset bcr ch4
        await eef.SetDigitalOutput(8,0) #reset bcr ch5

        led_bcr_ext_off = (await eef.GetDigitalInput(3))[0] #Test BCREXT Input

        PyLogger.logger.info(f"led_bcr_ext_on  : {led_bcr_ext_on}")
        PyLogger.logger.info(f"led_bcr_ext_off : {led_bcr_ext_off}")

        if led_bcr_ext_on != 1 or led_bcr_ext_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED BCREXT IO Test", log=True)
        else:
            await send_gc_msg(f"PASSED BCREXT IO Test", log=True)
        
        #Test M1, M2
        try:
            eef = get_node_endpoint('eef')
            await eef.Reset()
            await asyncio.sleep(0.1)
            await start_firmware('eef')
            corexy = get_dual_mover_endpoint('st')
            await corexy.SetProfile(
                handle=0, speed=50000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
            )
            await corexy.UseProfile(0)
            await corexy.SetParameter(DualMoverParameter.LinkModeA,                          2)
            await corexy.SetParameter(DualMoverParameter.LinkModeB,                          1)
            await corexy.SetParameter(DualMoverParameter.RampModeA,                          1)
            await corexy.SetParameter(DualMoverParameter.RampModeB,                          1)
            await corexy.SetParameter(DualMoverParameter.EncoderModeA,                       1)
            await corexy.SetParameter(DualMoverParameter.EncoderModeB,                       1)
            await corexy.SetParameter(DualMoverParameter.EncoderCorrectionFactorA,        12.8)
            await corexy.SetParameter(DualMoverParameter.EncoderCorrectionFactorB,        12.8)
            await corexy.SetParameter(DualMoverParameter.MaxEncoderDeviationA,             512)
            await corexy.SetParameter(DualMoverParameter.MaxEncoderDeviationB,             512)
            await corexy.SetConfigurationStatus(1)

            await corexy.SetPosition(0, 0)
            await corexy.Move(100000, 100000, timeout=5)
            await corexy.Move(     0, 200000, timeout=5)
            await corexy.Move(100000, 100000, timeout=5)
            await corexy.Move(     0,      0, timeout=5)

        except UrpcFirmwareException as ex:
            PyLogger.logger.exception('M1, M2 Test failed with', ex)
            await send_gc_msg(f"FAILED M1, M2 Test", log=True)
        else:
            await send_gc_msg(f"PASSED M1, M2 Test", log=True)

        #Test M3, M4
        try:
            await start_firmware('eef')

            eef = get_node_endpoint('eef')
            await eef.SetAnalogOutput(EEFAnalogOutput.PMT1TEC, 0.96)    #activate M3
            await asyncio.sleep(2)

            mover = get_mover_endpoint('as1')
            await mover.SetProfile(
                handle=0, speed=140000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
            )
            await mover.UseProfile(0)
            await mover.SetParameter(MoverParameter.MovementDirection, 0)
            await mover.SetConfigurationStatus(1)

            await mover.SetPosition(0)
            await corexy.SetPosition(0,0)
            await asyncio.sleep(.1)

            await asyncio.gather(
                mover.Move(200000, timeout=5),
                corexy.Move(100000, 100000, timeout=5)
            )

            await eef.SetAnalogOutput(EEFAnalogOutput.PMT1TEC, 0.0)    #deactivate M3

            await eef.Reset()
            await asyncio.sleep(.1)

            await start_firmware('eef')

            eef = get_node_endpoint('eef')
            corexy = get_dual_mover_endpoint('st')
            await corexy.SetProfile(
                handle=0, speed=50000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
            )
            await corexy.UseProfile(0)
            await corexy.SetParameter(DualMoverParameter.LinkModeA,                          2)
            await corexy.SetParameter(DualMoverParameter.LinkModeB,                          1)
            await corexy.SetParameter(DualMoverParameter.RampModeA,                          1)
            await corexy.SetParameter(DualMoverParameter.RampModeB,                          1)
            await corexy.SetParameter(DualMoverParameter.EncoderModeA,                       1)
            await corexy.SetParameter(DualMoverParameter.EncoderModeB,                       1)
            await corexy.SetParameter(DualMoverParameter.EncoderCorrectionFactorA,        12.8)
            await corexy.SetParameter(DualMoverParameter.EncoderCorrectionFactorB,        12.8)
            await corexy.SetParameter(DualMoverParameter.MaxEncoderDeviationA,            2048)
            await corexy.SetParameter(DualMoverParameter.MaxEncoderDeviationB,            2048)
            await corexy.SetConfigurationStatus(1)

            await corexy.SetPosition(0,0)
            await corexy.Move(100000, 100000, timeout=5)
            await asyncio.sleep(0.5)

            await eef.SetAnalogOutput(EEFAnalogOutput.PMT2TEC, 0.96)    #activate m4

            eef = get_node_endpoint('eef')
            mover = get_mover_endpoint('as2')
            await mover.SetProfile(
                handle=0, speed=140000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
            )
            await mover.UseProfile(0)
            await mover.SetParameter(MoverParameter.MovementDirection, 1)
            await mover.SetConfigurationStatus(1)

            await mover.SetPosition(0)
            await asyncio.sleep(0.5)

            await asyncio.gather(
                mover.Move(200000, timeout=5),
                corexy.Move(0, 200000, timeout=5)
            )
            await asyncio.sleep(0.5)

            await eef.SetAnalogOutput(EEFAnalogOutput.PMT2TEC, 0.0)    #deactivate m4

        except UrpcFirmwareException as ex:
            PyLogger.logger.exception('M3, M4 Test failed with', ex)
            await send_gc_msg(f"FAILED M3, M4 Test", log=True)
        else:
            await send_gc_msg(f"PASSED M3, M4 Test", log=True)

        # Test Alpha Laser Interface

        eef = get_node_endpoint('eef')
        await start_firmware('eef')
        await asyncio.sleep(.1)
        meas = get_measurement_endpoint()

        # Reset Power, Enable and Excitation -> Check PD as background offset
        await meas.SetParameter(MeasurementParameter.AlphaLaserPower, 0.0)
        await meas.SetParameter(MeasurementParameter.AlphaLaserEnable, 0)
        await set_alpha_excitation(0)
        background = (await eef.GetAnalogInput(EEFAnalogInput.ALPHAPHOTODIODE))[0]
        pd = 0

        PyLogger.logger.info(f"Alpha Laser Background: {background}")

        # Enable = 1, Excitation = 1, Power = 0.0 = 0 mA, 0.16 = 5.5 mA, 0.17 = 37.5 mA, 0.18 = 65 mA, 0.19 = 97 mA -> Check (PD - background)
        await meas.SetParameter(MeasurementParameter.AlphaLaserEnable, 1)
        await set_alpha_excitation(1)
        power_start_value = 0.13
        power_inkrement = 0.0001
        power_stop_value = 0.2
        power_05_value = 0
        power_value = power_start_value
        while power_value < power_stop_value:
            await meas.SetParameter(MeasurementParameter.AlphaLaserPower, power_value)
            pd = (await eef.GetAnalogInput(EEFAnalogInput.ALPHAPHOTODIODE))[0] - background
            if pd > 0.49 and pd < 0.52:
                power_05_value = power_value
            PyLogger.logger.info(f"ALD Power {power_value:.4f} : {pd:.4f}")
            power_value = power_value + power_inkrement

        PyLogger.logger.info(f"ALD Power = 0.5: {power_05_value:.4f}")
        
        if power_05_value < 0.135 or power_05_value > 0.19:
            eeftest_error = 1
            await send_gc_msg(f"FAILED ALD Excitation Test", log=True)
        else:
            await send_gc_msg(f"PASSED ALD Excitation Test", log=True)

        ALD_Temp_Error_off = (await eef.GetDigitalInput(EEFDigitalInput.ALPHATEMPERROR))[0]

        await eef.ConfigurePWMOutput(1,1000,1000,10)
        await eef.EnablePWMOutput(1,1)  #enable PWM4 for 5k6- resistor -> over temp error
        await asyncio.sleep(.1)

        ALD_Temp_Error_on = (await eef.GetDigitalInput(EEFDigitalInput.ALPHATEMPERROR))[0]

        PyLogger.logger.info(f"ALD_Temp_Error_off : {ALD_Temp_Error_off}")
        PyLogger.logger.info(f"ALD_Temp_Error_on  : {ALD_Temp_Error_on}")
    
        if ALD_Temp_Error_off != 0 or ALD_Temp_Error_on != 1:
            eeftest_error = 1
            await send_gc_msg(f"FAILED ALD Temp Test", log=True)
        else:
            await send_gc_msg(f"PASSED ALD Temp Test", log=True)

        await eef.EnablePWMOutput(1,0)  #disable PWM4 for 5k6- resistor -> over temp error

        #Test Alpha Tec
        # await start_firmware('eef')
        # eef = get_node_endpoint('eef')
        # await asyncio.sleep(.1)
        
        # await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.9)
        # await asyncio.sleep(15)
        # await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.0)

        #Test Interlock
        await start_firmware('eef')
        eef = get_node_endpoint('eef')

        ILStat_on = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCKSTATUS))[0]
        IL1_on = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCK1))[0]
        IL2_on = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCK2))[0]
        IL3_on = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCK3))[0]

        await eef.SetAnalogOutput(EEFAnalogOutput.RESERVETEC, 0.96)    #activate IL Relay
        await asyncio.sleep(1)

        ILStat_off = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCKSTATUS))[0]
        IL1_off = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCK1))[0]
        IL2_off = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCK2))[0]
        IL3_off = (await eef.GetDigitalInput(EEFDigitalInput.INTERLOCK3))[0]

        PyLogger.logger.info(f"ILStat_on  : {ILStat_on}")
        PyLogger.logger.info(f"IL1_on     : {IL1_on}")
        PyLogger.logger.info(f"IL2_on     : {IL2_on}")
        PyLogger.logger.info(f"IL3_on     : {IL3_on}")
        PyLogger.logger.info(f"ILStat_off : {ILStat_off}")
        PyLogger.logger.info(f"IL1_off    : {IL1_off}")
        PyLogger.logger.info(f"IL2_off    : {IL2_off}")
        PyLogger.logger.info(f"IL3_off    : {IL3_off}")
        
        if ILStat_on != 1 or IL1_on != 1 or IL2_on != 1 or IL3_on != 1 or ILStat_off != 0 or IL1_off != 0 or IL2_off != 0 or IL3_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED IL Test", log=True)
        else:
            await send_gc_msg(f"PASSED IL Test", log=True)

        await eef.SetAnalogOutput(EEFAnalogOutput.RESERVETEC, 0.0)    #deactivate IL Relay

        #TEST XFL

        await start_firmware('eef')
        meas = get_measurement_endpoint()
        fan = get_fan_control_endpoint('eef_fan')

        await fan.SetParameter(0, FanControlParameter.SpeedSampleTime, 100)

        await meas.SetParameter(MeasurementParameter.FlashLampPower, 1)
        await meas.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 1)
        await seq_set_signal('flash')

        await asyncio.sleep(.1)        

        await fan.SetSpeed(0, 100)
        await asyncio.sleep(2)
        pwm_freq_100_on = (await fan.GetSpeed(0))[0] / 60

        await fan.SetSpeed(0, 50)
        await asyncio.sleep(2)
        pwm_freq_50_on = (await fan.GetSpeed(0))[0] / 60

        await fan.SetSpeed(0, 0)
        await asyncio.sleep(2)
        pwm_freq_0_on = (await fan.GetSpeed(0))[0] / 60

        await meas.SetParameter(MeasurementParameter.FlashLampPower, 0)
        await meas.SetParameter(MeasurementParameter.FlashLampHighPowerEnable, 0)
        await seq_clear_signal('flash')

        await asyncio.sleep(.1)

        await fan.SetSpeed(0, 100)
        await asyncio.sleep(2)
        pwm_freq_100_off = (await fan.GetSpeed(0))[0] / 60

        await fan.SetSpeed(0, 50)
        await asyncio.sleep(2)
        pwm_freq_50_off = (await fan.GetSpeed(0))[0] / 60

        await fan.SetSpeed(0, 0)
        await asyncio.sleep(2)
        pwm_freq_0_off = (await fan.GetSpeed(0))[0] / 60

        PyLogger.logger.info(f"pwm_freq_0_on    : {pwm_freq_0_on:5.0f}")
        PyLogger.logger.info(f"pwm_freq_50_on   : {pwm_freq_50_on:5.0f}")
        PyLogger.logger.info(f"pwm_freq_100_on  : {pwm_freq_100_on:5.0f}")
        PyLogger.logger.info(f"pwm_freq_0_off   : {pwm_freq_0_off:5.0f}")
        PyLogger.logger.info(f"pwm_freq_50_off  : {pwm_freq_50_off:5.0f}")
        PyLogger.logger.info(f"pwm_freq_100_off : {pwm_freq_100_off:5.0f}")

        if pwm_freq_0_on != 0 or (pwm_freq_50_on < (25000 * 0.99)) or (pwm_freq_50_on > (25000 * 1.01)) or pwm_freq_100_on != 0 or pwm_freq_0_off != 0 or pwm_freq_50_off != 0 or pwm_freq_100_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED XFL Test", log=True)
        else:
            await send_gc_msg(f"PASSED XFL Test", log=True)

        #Test EXP
        await start_firmware('eef')
        eef = get_node_endpoint('eef')

        await eef.SetAnalogOutput(EEFAnalogOutput.PMT1TEC, 0.96)    #activate M3
        await eef.SetAnalogOutput(EEFAnalogOutput.PMT2TEC, 0.96)    #activate M4
        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.2)    #activate M3
        await asyncio.sleep(0.1)

        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.96)    #activate M3

        await eef.SetAnalogOutput(EEFAnalogOutput.RESERVETEC, 0.96)    #activate M3

        await asyncio.sleep(1)

        Ain0 = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN0))[0]*4.096*2*11
        Ain1 = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN1))[0]*4.096*2*11
        Ain2 = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN2))[0]*4.096*2*11
        Ain3 = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN3))[0]*4.096*2*11

        await eef.SetAnalogOutput(EEFAnalogOutput.PMT1TEC, 0)    #deactivate M3
        await eef.SetAnalogOutput(EEFAnalogOutput.PMT2TEC, 0)    #deactivate M4
        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0)    #deactivate M3
        await eef.SetAnalogOutput(EEFAnalogOutput.RESERVETEC, 0.)    #deactivate M3

        await eef.EnablePWMOutput(EEFPWMOutput.PWM1,0) #disable PWM on EXPAN PIN 3 for AIn2
        await eef.ConfigurePWMOutput(EEFPWMOutput.PWM1,1000,1000,10)
        await eef.EnablePWMOutput(EEFPWMOutput.PWM1,1) #enable PWM on EXPAN PIN 3 for AIn2
        await asyncio.sleep(.1)
        AIn2_EXP = 0
        AIn2_EXP = (await eef.GetAnalogInput(EEFAnalogInput.EXPAIN2))[0]
        await eef.EnablePWMOutput(EEFPWMOutput.PWM1,0)
        await eef.ConfigurePWMOutput(EEFPWMOutput.PWM1,0,0,10)

        await eef.EnablePWMOutput(EEFPWMOutput.PWM2,0) #disable PWM on EXPAN PIN 4 for AIn3
        await eef.ConfigurePWMOutput(EEFPWMOutput.PWM2,1000,1000,10)
        await eef.EnablePWMOutput(EEFPWMOutput.PWM2,1) #enable PWM on EXPAN PIN 4 for AIn3
        await asyncio.sleep(.1)
        AIn3_EXP = 0
        AIn3_EXP = (await eef.GetAnalogInput(EEFAnalogInput.EXPAIN3))[0]
        await eef.EnablePWMOutput(EEFPWMOutput.PWM2,0)
        await eef.ConfigurePWMOutput(EEFPWMOutput.PWM2,0,0,10)

        PyLogger.logger.info(f"EXP Ain0     : {Ain0:.2f}")
        PyLogger.logger.info(f"EXP Ain1     : {Ain1:.2f}")
        PyLogger.logger.info(f"EXP Ain2     : {Ain2:.2f}")
        PyLogger.logger.info(f"EXP Ain3     : {Ain3:.2f}")
        PyLogger.logger.info(f"EXP AIn2_EXP : {AIn2_EXP:.2f}")
        PyLogger.logger.info(f"EXP AIn3_EXP : {AIn3_EXP:.2f}")

        if ((Ain0 < 22) or (Ain0 > 23.5)) or ((Ain1 < 22) or (Ain1 > 23.5)) or ((Ain2 < 22) or (Ain2 > 23.5)) or ((Ain3 < 22) or (Ain3 > 23.5)) or ((AIn2_EXP < 0.4) or (AIn2_EXP > 0.6)) or ((AIn3_EXP < 0.4) or (AIn3_EXP > 0.6)):
            eeftest_error = 1
            await send_gc_msg(f"FAILED EXP Test", log=True)
        else:
            await send_gc_msg(f"PASSED EXP Test", log=True)

        #TEST REF Inputs

        await start_firmware('eef')
        eef = get_node_endpoint('eef')
        corexy = get_dual_mover_endpoint('st')
        mover3 = get_mover_endpoint('as1')
        mover4 = get_mover_endpoint('as2')

        await eef.SetDigitalOutput(EEFDigitalOutput.REFENABLE,1) #enable REFVoltage
        await asyncio.sleep(2)

        await corexy.SetParameter(DualMoverParameter.HomeSensorEnableA, 0x01)
        await corexy.SetParameter(DualMoverParameter.HomeSensorEnableB, 0x01)
        await mover3.SetParameter(MoverParameter.HomeSensorEnable, 0x01)
        await mover4.SetParameter(MoverParameter.HomeSensorEnable, 0x01)

        ref1_on, = await corexy.GetParameter(DualMoverParameter.HomeSensorStatusA)
        ref2_on, = await corexy.GetParameter(DualMoverParameter.HomeSensorStatusB)
        ref3_on, = await mover3.GetParameter(MoverParameter.HomeSensorStatus)
        ref4_on, = await mover4.GetParameter(MoverParameter.HomeSensorStatus)

        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.2)    #activate M3
        await asyncio.sleep(0.1)
        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.96)    #activate M3
        await asyncio.sleep(0.5)

        ref1_off, = await corexy.GetParameter(DualMoverParameter.HomeSensorStatusA)
        ref2_off, = await corexy.GetParameter(DualMoverParameter.HomeSensorStatusB)
        ref3_off, = await mover3.GetParameter(MoverParameter.HomeSensorStatus)
        ref4_off, = await mover4.GetParameter(MoverParameter.HomeSensorStatus)

        PyLogger.logger.info(f"ref1_on  : {ref1_on}")
        PyLogger.logger.info(f"ref2_on  : {ref2_on}")
        PyLogger.logger.info(f"ref3_on  : {ref3_on}")
        PyLogger.logger.info(f"ref4_on  : {ref4_on}")
        PyLogger.logger.info(f"ref1_off : {ref1_off}")
        PyLogger.logger.info(f"ref2_off : {ref2_off}")
        PyLogger.logger.info(f"ref3_off : {ref3_off}")
        PyLogger.logger.info(f"ref4_off : {ref4_off}")

        if ref1_on != 1 or ref2_on != 1 or ref3_on != 1 or ref4_on != 1 or ref1_off != 0 or ref2_off != 0 or ref3_off != 0 or ref4_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED REF Test", log=True)
        else:
            await send_gc_msg(f"PASSED REF Test", log=True)

        await eef.SetDigitalOutput(EEFDigitalOutput.REFENABLE,0) #disable REFVoltage

        #TEST USLFAN

        await start_firmware('eef')
        eef = get_node_endpoint('eef')
        meas = get_measurement_endpoint()
        fan = get_fan_control_endpoint('eef_fan')

        await meas.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, 0)

        await fan.SetParameter(1, FanControlParameter.SpeedSampleTime, 100)

        await fan.SetSpeed(1, 100)
        await asyncio.sleep(1)
        uslfan_0_off = (await fan.GetSpeed(1))[0] / 60

        await fan.SetSpeed(1, 50)
        await asyncio.sleep(1)
        uslfan_50_off = (await fan.GetSpeed(1))[0] / 60

        await fan.SetSpeed(1, 0)
        await asyncio.sleep(1)
        uslfan_100_off = (await fan.GetSpeed(1))[0] / 60

        await meas.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, 1)

        await fan.SetParameter(1, FanControlParameter.SpeedSampleTime, 100)

        await fan.SetSpeed(1, 100)
        await asyncio.sleep(1)
        uslfan_0_on = (await fan.GetSpeed(1))[0] / 60

        await fan.SetSpeed(1, 50)
        await asyncio.sleep(1)
        uslfan_50_on = (await fan.GetSpeed(1))[0] / 60

        await fan.SetSpeed(1, 0)
        await asyncio.sleep(1)
        uslfan_100_on = (await fan.GetSpeed(1))[0] / 60

        PyLogger.logger.info(f"uslfan_0_on    : {uslfan_0_on:5.0f}")
        PyLogger.logger.info(f"uslfan_50_on   : {uslfan_50_on:5.0f}")
        PyLogger.logger.info(f"uslfan_100_on  : {uslfan_100_on:5.0f}")
        PyLogger.logger.info(f"uslfan_0_off   : {uslfan_0_off:5.0f}")
        PyLogger.logger.info(f"uslfan_50_off  : {uslfan_50_off:5.0f}")
        PyLogger.logger.info(f"uslfan_100_off : {uslfan_100_off:5.0f}")

        if uslfan_0_on != 0 or (uslfan_50_on < (25000 * 0.99)) or (uslfan_50_on > (25000 * 1.01)) or uslfan_100_on != 0 or uslfan_0_off != 0 or uslfan_50_off != 0 or uslfan_100_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED USL Fan Test", log=True)
        else:
            await send_gc_msg(f"PASSED USL Fan Test", log=True)        
        
        #TEST TECs with Load + FAN1 + FAN2
        await start_firmware('eef')
        eef = get_node_endpoint('eef')
        await asyncio.sleep(.1)

        await eef.SetDigitalOutput(20,1) #enable TEC Load Relay
        await asyncio.sleep(1)

        await eef.SetAnalogOutput(EEFAnalogOutput.PMT1TEC, 0.4)    #TEC1
        await asyncio.sleep(0.5)
        Ain0_on = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN0))[0]*4.096*2*11
        await eef.SetAnalogOutput(EEFAnalogOutput.PMT1TEC, 0)    #TEC1
        await asyncio.sleep(0.5)
        Ain0_off = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN0))[0]*4.096*2*11

        await eef.SetAnalogOutput(EEFAnalogOutput.PMT2TEC, 0.4)    #TEC2
        await asyncio.sleep(0.5)
        Ain1_on = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN1))[0]*4.096*2*11
        await eef.SetAnalogOutput(EEFAnalogOutput.PMT2TEC, 0)    #TEC2
        await asyncio.sleep(0.5)
        Ain1_off = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN1))[0]*4.096*2*11

        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0.4)    #ALPHATEC
        await asyncio.sleep(0.5)
        Ain2_on = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN2))[0]*4.096*2*11
        await eef.SetAnalogOutput(EEFAnalogOutput.ALPHATEC, 0)    #ALPHATEC
        await asyncio.sleep(0.5)
        Ain2_off = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN2))[0]*4.096*2*11

        await eef.SetAnalogOutput(EEFAnalogOutput.RESERVETEC, 0.4)    #TECRES
        await asyncio.sleep(0.5)
        Ain3_on = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN3))[0]*4.096*2*11
        await eef.SetAnalogOutput(EEFAnalogOutput.RESERVETEC, 0)    #TECRES
        await asyncio.sleep(0.5)
        Ain3_off = (await eef.GetAnalogInput(EEFAnalogInput.TESTBOARDAIN3))[0]*4.096*2*11

        await eef.SetDigitalOutput(20,0) #disable TEC Load Relay

        PyLogger.logger.info(f"TEC/FAN1 Ain0_on  : {Ain0_on:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain1_on  : {Ain1_on:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain2_on  : {Ain2_on:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain3_on  : {Ain3_on:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain0_off : {Ain0_off:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain1_off : {Ain1_off:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain2_off : {Ain2_off:.0f}")
        PyLogger.logger.info(f"TEC/FAN1 Ain3_off : {Ain3_off:.0f}")

        if ((Ain0_on < 6) or (Ain0_on > 8)) or ((Ain1_on < 6) or (Ain1_on > 8)) or ((Ain2_on < 6) or (Ain2_on > 8)) or ((Ain3_on < 6) or (Ain3_on > 8)) or Ain0_off != 0 or Ain1_off != 0 or Ain2_off != 0 or Ain3_off != 0:
            eeftest_error = 1
            await send_gc_msg(f"FAILED TEC Test", log=True)
            await send_gc_msg(f"FAILED FAN1 Test", log=True)
        else:
            await send_gc_msg(f"PASSED TEC Test", log=True)
            await send_gc_msg(f"PASSED FAN1 Test", log=True)

        FAN2_24V = (await eef.GetAnalogInput(EEFAnalogInput.PMT3HIGHVOLTAGEMONITOR))[0]*2.5*3

        PyLogger.logger.info(f"FAN2_24V : {FAN2_24V:.2}")

        if (FAN2_24V < 4.5) or (FAN2_24V > 5.5):
            eeftest_error = 1
            await send_gc_msg(f"FAILED FAN2 Test", log=True)
        else:
            await send_gc_msg(f"PASSED FAN2 Test", log=True)

        #TEST Photodiodes

        await start_firmware('eef')
        eef = get_node_endpoint('eef')
        meas = get_measurement_endpoint()

        #Test PD1-REF
        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
        conversion_delay = 120000         #  12 us
        name = 'pd1'

        op_id = 'pd_measure'
        seq_gen = meas_seq_generator()
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # both low
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # reset high, range low
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # both high 
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # reset low, range high

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_autorange}) #both low -> 0V ->0
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})     #reset high, range low -> 1.65V -> 65535/2
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})          # both high -> 3V3 -> 65535
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=2)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_in_high_range})#reset low, range high -> 1.65V -> 65535/2
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=3)

        seq_gen.Stop(0)
        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 4)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"PD1 Results: {results}")

        if results[0] > 100 or ((results[1] < 30000) or (results[1] > 35000)) or results[2] < 64000 or ((results[3] < 30000) or (results[3] > 35000)):
            eeftest_error = 0
            await send_gc_msg(f"FAILED PD1 Test", log=True)
        else:
            await send_gc_msg(f"PASSED PD1 Test", log=True)

        #Test PD2-ABS
        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
        conversion_delay = 120000         #  12 us
        name = 'pd2'

        op_id = 'pd_measure'
        seq_gen = meas_seq_generator()
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # both low
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # reset high, range low
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # both high 
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # reset low, range high

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_autorange}) #both low -> 0V ->0
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})     #reset high, range low -> 1.65V -> 65535/2
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})          # both high -> 3V3 -> 65535
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=2)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_in_high_range})#reset low, range high -> 1.65V -> 65535/2
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=3)

        seq_gen.Stop(0)
        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 4)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"PD2 Results: {results}")

        if results[0] > 100 or ((results[1] < 30000) or (results[1] > 35000)) or results[2] < 64000 or ((results[3] < 30000) or (results[3] > 35000)):
            eeftest_error = 1
            await send_gc_msg(f"FAILED PD2 Test", log=True)
        else:
            await send_gc_msg(f"PASSED PD2 Test", log=True)

        #Test PD3-RES
        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
        conversion_delay = 120000         #  12 us
        name = 'pd3'

        op_id = 'pd_measure'
        seq_gen = meas_seq_generator()
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # both low
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # reset high, range low
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # both high 
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # reset low, range high

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_autorange}) #both low -> 0V ->0
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})     #reset high, range low -> 1.65V -> 65535/2
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})          # both high -> 3V3 -> 65535
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=2)

        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_in_high_range})#reset low, range high -> 1.65V -> 65535/2
        seq_gen.TimerWaitAndRestart(conversion_delay)
        seq_gen.SetTriggerOutput(pd_sample[name])
        seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=3)

        seq_gen.Stop(0)
        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 4)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"PD3 Results: {results}")

        if results[0] > 100 or ((results[1] < 30000) or (results[1] > 35000)) or results[2] < 64000 or ((results[3] < 30000) or (results[3] > 35000)):
            eeftest_error = 1
            await send_gc_msg(f"FAILED PD3 Test", log=True)
        else:
            await send_gc_msg(f"PASSED PD3 Test", log=True)

        #Test PD4
        if (await eef.GetHardwareRevision())[1] >= 8:
            meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
            conversion_delay = 120000         #  12 us
            name = 'pd4'

            op_id = 'pd_measure'
            seq_gen = meas_seq_generator()
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # both low
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # reset high, range low
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # both high
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # reset low, range high

            seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_autorange}) #both low -> 0V ->0
            seq_gen.TimerWaitAndRestart(conversion_delay)
            seq_gen.SetTriggerOutput(pd_sample[name])
            seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)

            seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})     #reset high, range low -> 1.65V -> 65535/2
            seq_gen.TimerWaitAndRestart(conversion_delay)
            seq_gen.SetTriggerOutput(pd_sample[name])
            seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)

            seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})          # both high -> 3V3 -> 65535
            seq_gen.TimerWaitAndRestart(conversion_delay)
            seq_gen.SetTriggerOutput(pd_sample[name])
            seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=2)

            seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_in_high_range})#reset low, range high -> 1.65V -> 65535/2
            seq_gen.TimerWaitAndRestart(conversion_delay)
            seq_gen.SetTriggerOutput(pd_sample[name])
            seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=3)

            seq_gen.Stop(0)
            meas_unit.ClearOperations()
            meas_unit.resultAddresses[op_id] = range(0, 4)
            await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            PyLogger.logger.info(f"PD4 Results: {results}")

            if results[0] > 100 or ((results[1] < 30000) or (results[1] > 35000)) or results[2] < 64000 or ((results[3] < 30000) or (results[3] > 35000)):
                eeftest_error = 1
                await send_gc_msg(f"FAILED PD4 Test", log=True)
            else:
                await send_gc_msg(f"PASSED PD4 Test", log=True)

        else:
            await send_gc_msg(f"Skipped PD4 Test", log=True)

        #TEST PMT1 IO, I2c and ADC

        await start_firmware('eef')
        eef = get_node_endpoint('eef')
        meas = get_measurement_endpoint()

        fixed_range         = 20000     

        seq_gen = meas_seq_generator()
        # Clear the result buffer
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)  # pmt1_1
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)  # pmt1_2
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)  # pmt1_3
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)  # pmt1_4
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)  # pmt1_5
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)  # pmt1_6
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)  # pmt1_7
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)  # pmt1_8
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)  # pmt1_9
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)  # pmt1_10
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt1_11
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt1_12
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt1_13
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt1_14
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt1_15
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt1_16

        #there are 4 signals that have to be switched in a test pattern:

        # Gating, HV Gate, RESET, Range
        #0:     Gating: 0, HV GATE: 0, RESET: 0, RANGE: 0 -> expected result: 0.000 V -> 0
        #1:     Gating: 0, HV GATE: 0, RESET: 0, RANGE: 1 -> expected result: 0.825 V -> 16383.75

        #2:     Gating: 0, HV GATE: 0, RESET: 1, RANGE: 0 -> expected result: 0.825 V -> 16383.75
        #3:     Gating: 0, HV GATE: 0, RESET: 1, RANGE: 1 -> expected result: 1.650 V -> 32767.5
        #4:     Gating: 0, HV GATE: 1, RESET: 0, RANGE: 0 -> expected result: 0.825 V -> 16383.75
        #5:     Gating: 0, HV GATE: 1, RESET: 0, RANGE: 1 -> expected result: 1.650 V -> 32767.5
        #6:     Gating: 0, HV GATE: 1, RESET: 1, RANGE: 0 -> expected result: 1.650 V -> 32767.5
        #7:     Gating: 0, HV GATE: 1, RESET: 1, RANGE: 1 -> expected result: 2.475 V -> 49151.25
        #8:     Gating: 1, HV GATE: 0, RESET: 0, RANGE: 0 -> expected result: 0.825 V -> 16383.75
        #9:     Gating: 1, HV GATE: 0, RESET: 0, RANGE: 1 -> expected result: 1.650 V -> 32767.5
        #10:     Gating: 1, HV GATE: 0, RESET: 1, RANGE: 0 -> expected result: 1.650 V -> 32767.5
        #11:     Gating: 1, HV GATE: 0, RESET: 1, RANGE: 1 -> expected result: 2.475 V -> 49151.25
        #12:     Gating: 1, HV GATE: 1, RESET: 0, RANGE: 0 -> expected result: 1.650 V -> 32767.5
        #13:     Gating: 1, HV GATE: 1, RESET: 0, RANGE: 1 -> expected result: 2.475 V -> 49151.25
        #14:     Gating: 1, HV GATE: 1, RESET: 1, RANGE: 0 -> expected result: 2.475 V -> 49151.25
        #15:     Gating: 1, HV GATE: 1, RESET: 1, RANGE: 1 -> expected result: 3.300 V -> 65535

        # Measure 1 
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
        seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 2
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 3
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=2)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 4 
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=3)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 5
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)
        seq_gen.SetSignals(OutputSignal.HVGatePMT1)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=4)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 6
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=5)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 7 
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=6)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 8
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=7)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 9
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)
        seq_gen.SetSignals(OutputSignal.InputGatePMT1)
        seq_gen.ResetSignals(OutputSignal.HVGatePMT1)   
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=8)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 10
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)        
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=9)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 11
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=10)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 12
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=11)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 13
        seq_gen.SetSignals(OutputSignal.HVGatePMT1)
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=12)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 14 
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=13)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 15
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=14)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 16
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=15)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Enable full reset, reset the offsets and disable HV gating
        seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
        seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
        seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)
        seq_gen.Stop(0)

        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

        op_id = 'pmt_measurement'
        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 16)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"PMT1 IO Results: {results}")

        if results[0] > 500 or ((results[1] < 15000) or (results[1] > 17000)) or ((results[2] < 15000) or (results[2] > 17000)) or ((results[3] < 32000) or (results[3] > 34000)) or ((results[4] < 15000) or (results[4] > 17000)) or ((results[5] < 32000) or (results[5] > 34000)) or ((results[6] < 32000) or (results[6] > 34000)) or ((results[7] < 48000) or (results[7] > 51000)) or ((results[8] < 15000) or (results[8] > 17000)) or ((results[9] < 32000) or (results[9] > 34000)) or ((results[10] < 32000) or (results[10] > 34000)) or ((results[11] < 48000) or (results[11] > 51000)) or ((results[12] < 32000) or (results[12] > 34000)) or ((results[13] < 48000) or (results[13] > 51000)) or ((results[14] < 48000) or (results[14] > 51000)) or results[15] < 65000:
            eeftest_error = 1
            await send_gc_msg(f"FAILED PMT1 IO Test", log=True)
        else:
            await send_gc_msg(f"PASSED PMT1 IO Test", log=True)

        PMT1HV = (await eef.GetAnalogInput(EEFAnalogInput.PMT1HIGHVOLTAGEMONITOR))[0]*3.3*2

        PyLogger.logger.info(f"PMT1HV: {PMT1HV:.2f}")

        if PMT1HV < 3.1 or PMT1HV > 3.5:
            eeftest_error = 1
            await send_gc_msg(f"FAILED PMT1 I2C Test", log=True)
        else:
            await send_gc_msg(f"PASSED PMT1 I2C Test", log=True)

        #TEST PMT2 IO, I2c and ADC

        eef = get_node_endpoint('eef')

        fixed_range         = 20000     

        seq_gen = meas_seq_generator()
        # Clear the result buffer
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)  # pmt2_1
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)  # pmt2_2
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)  # pmt2_3
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)  # pmt2_4
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=4)  # pmt2_5
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=5)  # pmt2_6
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=6)  # pmt2_7
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=7)  # pmt2_8
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=8)  # pmt2_9
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=9)  # pmt2_10
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_11
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_12
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt2_13
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt2_14
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt2_15
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt2_16

        #there are 4 signals that have to be switched in a test pattern:

        # Gating, HV Gate, RESET, Range
        #0:     Gating: 0, HV GATE: 0, RESET: 0, RANGE: 0 -> expected result: 0.000 V -> 0
        #1:     Gating: 0, HV GATE: 0, RESET: 0, RANGE: 1 -> expected result: 0.825 V -> 16383.75

        #2:     Gating: 0, HV GATE: 0, RESET: 1, RANGE: 0 -> expected result: 0.825 V -> 16383.75
        #3:     Gating: 0, HV GATE: 0, RESET: 1, RANGE: 1 -> expected result: 1.650 V -> 32767.5
        #4:     Gating: 0, HV GATE: 1, RESET: 0, RANGE: 0 -> expected result: 0.825 V -> 16383.75
        #5:     Gating: 0, HV GATE: 1, RESET: 0, RANGE: 1 -> expected result: 1.650 V -> 32767.5
        #6:     Gating: 0, HV GATE: 1, RESET: 1, RANGE: 0 -> expected result: 1.650 V -> 32767.5
        #7:     Gating: 0, HV GATE: 1, RESET: 1, RANGE: 1 -> expected result: 2.475 V -> 49151.25
        #8:     Gating: 1, HV GATE: 0, RESET: 0, RANGE: 0 -> expected result: 0.825 V -> 16383.75
        #9:     Gating: 1, HV GATE: 0, RESET: 0, RANGE: 1 -> expected result: 1.650 V -> 32767.5
        #10:     Gating: 1, HV GATE: 0, RESET: 1, RANGE: 0 -> expected result: 1.650 V -> 32767.5
        #11:     Gating: 1, HV GATE: 0, RESET: 1, RANGE: 1 -> expected result: 2.475 V -> 49151.25
        #12:     Gating: 1, HV GATE: 1, RESET: 0, RANGE: 0 -> expected result: 1.650 V -> 32767.5
        #13:     Gating: 1, HV GATE: 1, RESET: 0, RANGE: 1 -> expected result: 2.475 V -> 49151.25
        #14:     Gating: 1, HV GATE: 1, RESET: 1, RANGE: 0 -> expected result: 2.475 V -> 49151.25
        #15:     Gating: 1, HV GATE: 1, RESET: 1, RANGE: 1 -> expected result: 3.300 V -> 65535

        # Measure 1 
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.ResetSignals(OutputSignal.HVGatePMT2)
        seq_gen.ResetSignals(OutputSignal.InputGatePMT2)
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_low_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 2
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.low_range_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=1)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 3
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=2)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 4 
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=3)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 5
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_low_range)
        seq_gen.SetSignals(OutputSignal.HVGatePMT2)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=4)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 6
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.low_range_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=5)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 7 
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=6)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 8
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=7)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 9
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_low_range)
        seq_gen.SetSignals(OutputSignal.InputGatePMT2)
        seq_gen.ResetSignals(OutputSignal.HVGatePMT2)   
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=8)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 10
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.low_range_reset)        
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=9)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 11
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=10)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 12
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=11)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 13
        seq_gen.SetSignals(OutputSignal.HVGatePMT2)
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_low_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=12)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 14 
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.low_range_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=13)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 15
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_high_range)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=14)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Measure 16
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.full_reset)
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT2)
        seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=False, dword=False, addrPos=0, resultPos=15)
        seq_gen.TimerWaitAndRestart(fixed_range)

        # Enable full reset, reset the offsets and disable HV gating
        seq_gen.ResetSignals(OutputSignal.HVGatePMT2)
        seq_gen.ResetSignals(OutputSignal.InputGatePMT2)
        seq_gen.SetIntegratorMode(pmt2=IntegratorMode.integrate_in_low_range)
        seq_gen.Stop(0)

        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

        op_id = 'pmt_measurement'
        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 16)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"PMT2 IO Results: {results}")

        if results[0] > 500 or ((results[1] < 15000) or (results[1] > 17000)) or ((results[2] < 15000) or (results[2] > 17000)) or ((results[3] < 32000) or (results[3] > 34000)) or ((results[4] < 15000) or (results[4] > 17000)) or ((results[5] < 32000) or (results[5] > 34000)) or ((results[6] < 32000) or (results[6] > 34000)) or ((results[7] < 48000) or (results[7] > 51000)) or ((results[8] < 15000) or (results[8] > 17000)) or ((results[9] < 32000) or (results[9] > 34000)) or ((results[10] < 32000) or (results[10] > 34000)) or ((results[11] < 48000) or (results[11] > 51000)) or ((results[12] < 32000) or (results[12] > 34000)) or ((results[13] < 48000) or (results[13] > 51000)) or ((results[14] < 48000) or (results[14] > 51000)) or results[15] < 65000:
            eeftest_error = 1
            await send_gc_msg(f"FAILED PMT2 IO Test", log=True)
        else:
            await send_gc_msg(f"PASSED PMT2 IO Test", log=True)

        PMT2HV = (await eef.GetAnalogInput(EEFAnalogInput.PMT2HIGHVOLTAGEMONITOR))[0]*3.3*2

        PyLogger.logger.info(f"PMT2HV: {PMT2HV:.2f}")

        if PMT2HV < 3.1 or PMT2HV > 3.5:
            eeftest_error = 1
            await send_gc_msg(f"FAILED PMT2 I2C Test", log=True)
        else:
            await send_gc_msg(f"PASSED PMT2 I2C Test", log=True)

        #TEST PMT1, PMT2, USL counting

        window_ms=100.0

        if (window_ms < 0.001):
            raise ValueError(f"window_ms must be greater or equal to 0.001 ms")

        window_us = round(window_ms * 1000)

        window_coarse, window_fine = divmod(window_us, 65536)

        hv_gate_delay       = 1000000   #  10 ms
        full_reset_delay    = 40000     # 400 us
        conversion_delay    = 1200      #  12 us
        switch_delay        = 25        # 250 ns
        gating_delay        = 100       #   1 us
        pre_cnt_window      = 100       #   1 us
        fixed_range         = 2000      #  20 us

        seq_gen = meas_seq_generator()
        # Clear the result buffer
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)  # pmt1_cr
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)  # pmt2_cr
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)  # pmt3_cr

        # Start counting
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT3, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
        if window_coarse > 0:
            seq_gen.Loop(window_coarse)
            seq_gen.Loop(65536)
            seq_gen.TimerWaitAndRestart(pre_cnt_window)
            seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.PulseCounterControl(MeasurementChannel.PMT3, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.LoopEnd()
            seq_gen.LoopEnd()
        if window_fine > 0:
            seq_gen.Loop(window_fine)
            seq_gen.TimerWaitAndRestart(pre_cnt_window)
            seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.PulseCounterControl(MeasurementChannel.PMT3, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.LoopEnd()

        # Save the counting results
        seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
        seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=1)
        seq_gen.GetPulseCounterResult(MeasurementChannel.PMT3, deadTime=False, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=2)
        seq_gen.Stop(0)

        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

        op_id = 'pmt_measurement'
        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 3)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        PyLogger.logger.info(f"pmt1_cr: {results[0]}")
        PyLogger.logger.info(f"pmt2_cr: {results[1]}")
        PyLogger.logger.info(f"pmt3_cr: {results[2]}")

        if  ((results[0] < 24998000) or (results[0] > 25002000)) or ((results[1] < 24998000) or (results[1] > 25002000)) or ((results[2] < 24998000) or (results[2] > 25002000)):
            eeftest_error = 1
            await send_gc_msg(f"FAILED PMT1, PMT2, USL Counting Test", log=True)
        else:
            await send_gc_msg(f"PASSED PMT1, PMT2, USL Counting Test", log=True)

        #Test USL HTS Alpha power and excitation
        await start_firmware('eef')
        eef = get_node_endpoint('eef')
        meas = get_measurement_endpoint()
        await asyncio.sleep(0.5)

        conversion_delay  = 12000    #  120 us

        await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 1)

        meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
        op_id = 'hts_alpha_on'
        seq_gen = meas_seq_generator()
        seq_gen.SetSignals(OUTPUTSIGNAL_ALPHA_HTS)
        seq_gen.Stop(0)
        meas_unit.ClearOperations()
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)

        Excitation = (await eef.GetAnalogInput(EEFAnalogInput.HTSALPHAPHOTODIODE))[0]*2.5

        EEFpwr = (await eef.GetAnalogInput(EEFAnalogInput.HTSALPHATEMPIN))[0]*2.5*3

        await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 0)

        PyLogger.logger.info(f"HTS Alpha Power: {EEFpwr}")

        if (EEFpwr < 3.0) or (EEFpwr > 3.6):
            eeftest_error = 1
            await send_gc_msg(f"FAILED HTS Alpha Power Test", log=True)
        else:
            await send_gc_msg(f"PASSED HTS Alpha Power Test", log=True)

        PyLogger.logger.info(f"HTS Alpha Excitation: {Excitation}")

        if (Excitation < 1.3) or (Excitation > 1.7):
            eeftest_error = 1
            await send_gc_msg(f"FAILED HTS Alpha Excitation Test", log=True)
        else:
            await send_gc_msg(f"PASSED HTS Alpha Excitation Test", log=True)

        if eeftest_error == 0:
            return f"EEF test successful. Continue with next DUT."
        else:
            return f"EEF test failed. Check error messages."

    finally:
        # By using a try/finally the power is always switched off when leaving the test function.
        await fmb.SetDigitalOutput(FMBDigitalOutput.PSUON, 1)   # Base Tester Power = OFF (active low)
        await asyncio.sleep(0.1)


async def eef_fpga_programming_script():

    #Programming script for EEF FPGA over CAN interface

    fmb = get_node_endpoint('fmb')

    # Reset the ground control stop button status at the beginning of the test.
    GlobalVar.set_stop_gc(False)
    # Send information and test results to ground control, that will be printed in the output text box.

    await start_firmware('fmb')
    await fmb.SetDigitalOutput(FMBDigitalOutput.PSUON, 0)       # Base Tester Power = ON (active low)

    await send_gc_msg(f"Start Programming EEF FPGA. This takes aprox. 5 minutes", log=True)

    try:
        await asyncio.sleep(1)
        fw_info = await update_fpga('/home/operator/.PyRInstall/mots/EEF_FPGA.mot')
        if fw_info == 'update_fpga() done':
            await send_gc_msg(f"Programming of FPGA SUCCESSFULL. Continue with next Device.", log=True)
        else:
            await send_gc_msg(f"Programming of FPGA FAILED", log=True)

    finally:
        # By using a try/finally the power is always switched off when leaving the test function.
        await fmb.SetDigitalOutput(FMBDigitalOutput.PSUON, 1)   # Base Tester Power = OFF (active low)
        await asyncio.sleep(0.1)
